/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.events.*;

public class List extends Scrollable {

public List (Composite parent, int style) {
	super (parent, checkStyle (style));
}

public void add (String string) {
	add (string, getItemCount());
}

public void add (String string, int index) {
	checkWidget();
	if (string == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (!(0 <= index && index <= getItemCount())) {
		error (SWT.ERROR_INVALID_RANGE);
	}
	_add (string, index);
}

public void addSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener (listener);
	addListener (SWT.Selection,typedListener);
	addListener (SWT.DefaultSelection,typedListener);
}

static int checkStyle (int style) {
	return checkBits (style, SWT.SINGLE, SWT.MULTI, 0, 0, 0, 0);
}

public Point computeSize (int wHint, int hHint, boolean changed) {
	checkWidget ();
	Point point = super.computeSize(wHint, hHint, changed);
	Rectangle trim = computeTrim (0, 0, point.x, point.y);
	return new Point (trim.width, trim.height);
}

public void deselect (int[] indices) {
	checkWidget();
	if (indices == null) error (SWT.ERROR_NULL_ARGUMENT);
	int count = getItemCount();
	for (int i=0; i<indices.length; i++){
		int index = indices [i];
		if (index < 0 || index > count - 1) continue;
		deselect(indices[i]);
	}
}

public void deselect (int index) {
	checkWidget();
	if (!(0 <= index && index < getItemCount()))  return;
	_deselect (index);
}

public void deselect (int start, int end) {
	checkWidget();
	if (start < 0 && end < 0) return;
	int count = getItemCount();
	if (start >= count && end >= count) return;
	for (int i=start; i<end; i++){
		deselect(i);
	}
}

public void deselectAll () {
	checkWidget();
	for (int i=0; i<getItemCount(); i++){
		deselect(i);
	}
}

public int getFocusIndex () {
	return _getFocusIndex ();
}

public String getItem (int index) {
	checkWidget();
	if (!(0 <= index && index < getItemCount()))  {
		error (SWT.ERROR_INVALID_RANGE);
	}
	return _getItem (index);
}

public int getItemCount () {
	checkWidget();
	return _getItemCount();
}

public int getItemHeight () {
	checkWidget();
	return _getItemHeight ();
}

public String[] getItems () {
	checkWidget();
	String [] string = new String[getItemCount()];
	for (int i=0; i<getItemCount(); i++){
		string[i] = getItem(i);
	}
	return string;
}

public String[] getSelection () {
	checkWidget();
	String [] selection = new String[getSelectionCount()];
	System.arraycopy(_getSelection(), 0, selection, 0, selection.length);
	return selection;
}

public int getSelectionCount () {
	checkWidget();
	return _getSelectionCount ();
}

public int getSelectionIndex () {
	checkWidget();
	if (getSelectionCount()==0) { return -1; }
	if (isSelected (getFocusIndex())) {
		return getFocusIndex();
	}
	return getSelectionIndices()[0];
}

public int[] getSelectionIndices () {
	return _getSelectionIndices ();
}

//TODO
public int getTopIndex () {
	return 0;
}

void hookEvents () {
	super.hookEvents();
	_hookMouseSelection("onclick");
	_hookKeySelection("onkeydown");
}

public int indexOf (String string) {
	checkWidget();
	return indexOf (string, 0);
}

public int indexOf (String string, int start) {
	checkWidget();
	if (string == null) error (SWT.ERROR_NULL_ARGUMENT);
	String [] items = getItems ();
	for (int i=start; i<items.length; i++) {
		if (items [i].equals (string)) return i;
	}
	return -1;
}

public boolean isSelected (int index) {
	checkWidget();
	return _isSelected (index);
}

public void remove (int[] indices) {
	checkWidget();
	if (indices == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (indices.length == 0) return;
	int [] newIndices = new int [indices.length];
	System.arraycopy (indices, 0, newIndices, 0, indices.length);
	sort (newIndices);
	int start = newIndices [newIndices.length - 1], end = newIndices [0];
	int count = getItemCount();
	if (!(0 <= start && start <= end && end < count)) {
		error (SWT.ERROR_INVALID_RANGE);
	}
	for (int i=0; i<newIndices.length; i++){
		remove(newIndices[i]);
	}
}

public void remove (int index) {
	checkWidget();
	_remove (index);
}

public void remove (int start, int end) {
	checkWidget();
	if (start > end) return;
	if (!(0 <= start && start <= end && end < getItemCount())) {
		error (SWT.ERROR_INVALID_RANGE);
	}
	for (int i=end; i>=start; i--) {
		remove(i);
	}
}

public void remove (String string) {
	checkWidget();
	if (string == null) error (SWT.ERROR_NULL_ARGUMENT);
	int index = indexOf (string, 0);
	if (index == -1) error (SWT.ERROR_INVALID_ARGUMENT);
	remove (index);
}

public void removeAll () {
	checkWidget();
	_removeAll();
}

public void removeSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Selection, listener);
	eventTable.unhook (SWT.DefaultSelection,listener);
}

public void select (int[] indices) {
	checkWidget();
	if (indices == null) error (SWT.ERROR_NULL_ARGUMENT);
	int length = indices.length;
	if (length == 0 || ((style & SWT.SINGLE) != 0 && length > 1)) return;
	for (int i=0; i<indices.length; i++){
		select(indices[i]);
	}
}

public void select (int index) {
	checkWidget();
	if (!(0 <= index && index < getItemCount()))  return;
	_select (index);
}

public void select (int start, int end) {
	checkWidget ();
	if (end < 0 || start > end || ((style & SWT.SINGLE) != 0 && start != end)) return;
	int count = getItemCount();
	if (count == 0 || start >= count) return;
	start = Math.max (0, start);
	end = Math.min (end, count - 1);
	for (int i=start; i<=end; i++){
		select(i);
	}
}

public void selectAll () {
	checkWidget();
	if ((style & SWT.SINGLE) != 0) return;
	for (int i=0; i<getItemCount(); i++){
		select(i);
	}
}

//TODO
public void setFont (Font font) {
}

void setFocusIndex (int index) {
	checkWidget ();	
	_setFocusIndex (index);
}

public void setItem (int index, String string) {
	checkWidget();
	if (string == null) error (SWT.ERROR_NULL_ARGUMENT);
	_setItem (index, string);
}

public void setItems (String[] items) {
	checkWidget();
	if (items == null) error (SWT.ERROR_NULL_ARGUMENT);
	removeAll();
	for (int i=0; i<items.length; i++) {
		add(items[i]);
	}
}

public void setSelection (int[] indices) {
	checkWidget ();
	if (indices == null) error (SWT.ERROR_NULL_ARGUMENT);
	deselectAll ();
	int length = indices.length;
	if (length == 0 || ((style & SWT.SINGLE) != 0 && length > 1)) return;
	if ((style & SWT.MULTI) != 0) {
		select (indices);
	}
	showSelection ();
}

public void setSelection (String[] items) {
	checkWidget ();
	if (items == null) error (SWT.ERROR_NULL_ARGUMENT);
	deselectAll ();
	int length = items.length;
	if (length == 0 || ((style & SWT.SINGLE) != 0 && length > 1)) return;
	boolean first = true;
	for (int i = 0; i < length; i++) {
		int index = 0;
		String string = items [i];
		if (string != null) {
			while ((index = indexOf (string, index)) != -1) {
				if ((style & SWT.MULTI) != 0) {
					if (first) {
						first = false;
//						selectFocusIndex (index);
					} else {
						select (index);
					}
				} else {
//					selectFocusIndex (index);
					break;
				}
				index++;
			}
		}
	}
	showSelection ();
}

public void setSelection (int index) {
	checkWidget ();
	deselectAll ();
	select (index);
	if ((style & SWT.MULTI) != 0) {
		if (index >= 0) setFocusIndex (index);
	}
}

public void setSelection (int start, int end) {
	checkWidget ();
	deselectAll ();
	if (end < 0 || start > end || ((style & SWT.SINGLE) != 0 && start != end)) return;
	int count = getItemCount();
	if (count == 0 || start >= count) return;
	start = Math.max (0, start);
	end = Math.min (end, count - 1);
	if ((style & SWT.MULTI) != 0) {
		select (start, end);
		setFocusIndex (start);
	}
	showSelection ();
}

//TODO
public void setTopIndex (int index) {
}
//TODO
public void showSelection () {
}

/*---------------------- NATIVE INTERFACE -------------------*/

protected native void _createHandle (Widget parent, int style, int index) /*-{
	if(!$wnd.dojo._hasResource["org.eclipse.swt.List"]){
		$wnd.dojo._hasResource["org.eclipse.swt.List"] = true;
		$wnd.dojo.provide("org.eclipse.swt.List");
		
		$wnd.dojo.require("dijit.layout._LayoutWidget");
		$wnd.dojo.require("dijit._Templated");
		
		$wnd.dojo.declare("org.eclipse.swt.List", $wnd.eval("[dijit.layout._LayoutWidget, dijit._Templated]"), {
			baseClass: "swtList",
			border: "",
			templateString: 
				'<div class="${baseClass}" baseClass="${baseClass}" id="${id}">' +
					'<div dojoAttachPoint="focusNode">' +
						'<div dojoAttachPoint="containerNode" class="${baseClass}Body"></div>' +
					'</div>' +
				'</div>',
			addItem: function(name, index){
				if (index >= -1 && index <= this.containerNode.childNodes.length) {
					var node = $doc.createElement("div");
					node.innerHTML = name;
					node.setAttribute("tabindex", 0);
					node.setAttribute("selected", false);
					node.className = "swtListItem";
					node.id = $wnd.dijit.getUniqueId("ListItem");
					if (index==-1 || this.containerNode.childNodes.length == index) {
						this.containerNode.appendChild(node);
					} else {
						this.containerNode.insertBefore(node,this.containerNode.childNodes[index]);
					}
				} else {
//TODO throw real error					
					$wnd.console.debug("Error adding list item");
				}
			},
			deselect: function(index){
				var item = this.getItem(index); 
				if(item){
					item.selected = false;
					$wnd.dojo.removeClass(this.containerNode.childNodes[index],"swtListItemSelected");
					$wnd.dijit.focus(item);
				}
			},
			getItem: function(index) {
				return this.containerNode.childNodes[index] ? this.containerNode.childNodes[index] : null;				
			},
			getItemValue: function(index) {
				return this.getItem(index) ? this.getItem(index).innerHTML : null;				
			},
			indexOf: function(item) {
				return $wnd.dojo.indexOf(this.containerNode.childNodes, item, 0, false);
			}, 
			isSelected: function(index) {
				return this.getItem(index) ? this.getItem(index).selected : false;
			},
			postCreate: function(){
				$wnd.swt.setCssStyle(this.domNode, {
					border: this.border,
					overflow: "hidden"
				});
			},
			remove: function(index){
				if(this.getItem(index)) {
					this.containerNode.removeChild(getItem(index));
				}
			},
			removeAll: function(){
				this.containerNode.innerHTML = ""; 
			},
			resize: function(dim){},
			select: function(index){
				var item = this.getItem(index); 
				if(item){
					item.selected = true;
					$wnd.dojo.addClass(this.containerNode.childNodes[index],"swtListItemSelected");
					$wnd.dijit.focus(item);
				}
			},
			setItemValue: function(index, string){
				if(this.getItem(index)) {
					this.getItem(index).innerHTML = string;
				}
			}
		});
	}
	
	var jsParent = parent.@org.eclipse.swt.widgets.Widget::jsObject;
	params = {};
	if((style & (@org.eclipse.swt.SWT::BORDER)) != 0){
		params.border = "1px solid black";
	}
	var self = new $wnd.org.eclipse.swt.List(params);
	try {
		this.@org.eclipse.swt.widgets.Widget::jsObject = self;
		jsParent.addChild(self);
	} catch (e) {
//TODO Have to throw real exception for Java side also	
		$wnd.console.log(e);
	}
}-*/;

native void _add (String string, int index) /*-{
 	this.@org.eclipse.swt.widgets.Widget::jsObject.addItem(string, index);
}-*/;

native void _deselect (int index) /*-{
	this.@org.eclipse.swt.widgets.Widget::jsObject.deselect(index);
}-*/;

native int _getFocusIndex () /*-{
	return this.@org.eclipse.swt.widgets.Widget::jsObject.indexOf( 
		$wnd.dijit.getFocus(
			this.@org.eclipse.swt.widgets.Widget::jsObject.containerNode.childNodes
		).node
	)
}-*/;

native String _getItem (int index) /*-{
	return this.@org.eclipse.swt.widgets.Widget::jsObject.getItemValue(index);
}-*/;

native int _getItemCount () /*-{
	return this.@org.eclipse.swt.widgets.Widget::jsObject.containerNode.childNodes.length;
}-*/;

native int _getItemHeight () /*-{
	var self = this.@org.eclipse.swt.widgets.Widget::jsObject;
	return self.containerNode.firstChild ? $wnd.swt.getNativeBounds(self.containerNode.firstChild).h : 0;
}-*/;

native String [] _getSelection () /*-{
	var array = $wnd.dojo.query(".swtListItemSelected",this.@org.eclipse.swt.widgets.Widget::jsObject.id);
	for (var i=0;i<array.length;i++) {array[i] = array[i].innerHTML;}
	return array;
}-*/;

native int [] _getSelectionIndices () /*-{
	var self = this.@org.eclipse.swt.widgets.Widget::jsObject;
	var indices = [];
	for (var i=0; i< self.containerNode.childNodes.length; i++) {
		if (self.isSelected(i)) {
			indices.push(i);
		}
	}
	return indices;
}-*/;

native int _getSelectionCount () /*-{
	return $wnd.dojo.query(".swtListItemSelected",this.@org.eclipse.swt.widgets.Widget::jsObject.id).length;
}-*/;

native void _hookKeySelection (String eventType) /*-{
	var self = this;
	$wnd.dojo.connect(
		self.@org.eclipse.swt.widgets.Widget::jsObject.domNode,
		eventType,
		function(arg){
			if (arg) {
				arg.preventDefault();
				var index = self.@org.eclipse.swt.widgets.Widget::jsObject.indexOf(arg.target);
				if((self.@org.eclipse.swt.widgets.Widget::style & (@org.eclipse.swt.SWT::MULTI)) != 0) {
					if (arg.shiftKey) {
						var focusIndex = self.@org.eclipse.swt.widgets.List::getFocusIndex()();
						if (focusIndex >= 0) {
							switch(arg.keyCode){
								case $wnd.dojo.keys.DOWN_ARROW:
								case $wnd.dojo.keys.RIGHT_ARROW:
									if (self.@org.eclipse.swt.widgets.List::isSelected(I)(index+1)) {
										self.@org.eclipse.swt.widgets.List::deselect(I)(index+1);
									} else {
										self.@org.eclipse.swt.widgets.List::select(I)(index+1);
									} break;
								case $wnd.dojo.keys.UP_ARROW:
								case $wnd.dojo.keys.LEFT_ARROW:
									if (self.@org.eclipse.swt.widgets.List::isSelected(I)(index-1)) {
										self.@org.eclipse.swt.widgets.List::deselect(I)(index-1);
									} else {
										self.@org.eclipse.swt.widgets.List::select(I)(index-1);
									} break;
							}
					 	} else {
							switch(arg.keyCode){
								case $wnd.dojo.keys.DOWN_ARROW:
								case $wnd.dojo.keys.RIGHT_ARROW:
									self.@org.eclipse.swt.widgets.List::setFocusIndex(I)(index+1); break;
								case $wnd.dojo.keys.UP_ARROW:
								case $wnd.dojo.keys.LEFT_ARROW:
									self.@org.eclipse.swt.widgets.List::setFocusIndex(I)(index-1); break;
							}
						}	
					} else if (arg.ctrlKey) {						
						switch(arg.keyCode){
							case $wnd.dojo.keys.DOWN_ARROW:
							case $wnd.dojo.keys.RIGHT_ARROW:
								self.@org.eclipse.swt.widgets.List::setFocusIndex(I)(index+1); break;
							case $wnd.dojo.keys.UP_ARROW:
							case $wnd.dojo.keys.LEFT_ARROW:
								self.@org.eclipse.swt.widgets.List::setFocusIndex(I)(index-1); break;
						}
					} else {
						switch(arg.keyCode){
							case $wnd.dojo.keys.DOWN_ARROW:
							case $wnd.dojo.keys.RIGHT_ARROW:
								self.@org.eclipse.swt.widgets.List::setSelection(I)(index+1); break;
							case $wnd.dojo.keys.UP_ARROW:
							case $wnd.dojo.keys.LEFT_ARROW:
								self.@org.eclipse.swt.widgets.List::setSelection(I)(index-1); break;
						}
					} 
				
				} else if((self.@org.eclipse.swt.widgets.Widget::style & (@org.eclipse.swt.SWT::SINGLE)) != 0) {
					switch(arg.keyCode){
						case $wnd.dojo.keys.DOWN_ARROW:
						case $wnd.dojo.keys.RIGHT_ARROW:
							self.@org.eclipse.swt.widgets.List::setSelection(I)(index+1); break;
						case $wnd.dojo.keys.UP_ARROW:
						case $wnd.dojo.keys.LEFT_ARROW:
							self.@org.eclipse.swt.widgets.List::setSelection(I)(index-1); break;
					}
				}
			}
		}
	);
}-*/;

native void _hookMouseSelection (String eventType) /*-{
	var self = this;
	$wnd.dojo.connect(
		self.@org.eclipse.swt.widgets.Widget::jsObject.domNode,
		eventType,
		function(arg){
			if (arg) {
				arg.preventDefault();
				var index = self.@org.eclipse.swt.widgets.Widget::jsObject.indexOf(arg.target);
				if((self.@org.eclipse.swt.widgets.Widget::style & (@org.eclipse.swt.SWT::MULTI)) != 0) {
					if (arg.shiftKey) {
						var focusIndex = self.@org.eclipse.swt.widgets.List::getFocusIndex()();
						if (focusIndex >= 0) {
							self.@org.eclipse.swt.widgets.List::setSelection(II)(
								Math.min(focusIndex,index),
								Math.max(focusIndex,index)
							);
						} else {
							self.@org.eclipse.swt.widgets.List::setSelection(I)(index);
						}
					} else if (arg.ctrlKey) {
						if (self.@org.eclipse.swt.widgets.List::isSelected(I)(index)) {
							self.@org.eclipse.swt.widgets.List::deselect(I)(index);
						} else {
							self.@org.eclipse.swt.widgets.List::select(I)(index);
						}
					} else {
						self.@org.eclipse.swt.widgets.List::setSelection(I)(index);
					} 
				
				} else if((self.@org.eclipse.swt.widgets.Widget::style & (@org.eclipse.swt.SWT::SINGLE)) != 0) {
					self.@org.eclipse.swt.widgets.List::setSelection(I)(index);
				}
			}
		}
	);
}-*/;

native boolean _isSelected (int index) /*-{
	return this.@org.eclipse.swt.widgets.Widget::jsObject.isSelected(index);
}-*/;

native void _remove (int index) /*-{
	this.@org.eclipse.swt.widgets.Widget::jsObject.remove(index);
}-*/;

native void _removeAll () /*-{
	this.@org.eclipse.swt.widgets.Widget::jsObject.removeAll();
}-*/;

native void _select (int index) /*-{
	this.@org.eclipse.swt.widgets.Widget::jsObject.select(index);
}-*/;

native void _setFocusIndex (int index) /*-{
	$wnd.dijit.focus(this.@org.eclipse.swt.widgets.Widget::jsObject.getItem(index));
}-*/;

native void _setItem (int index, String string) /*-{
	this.@org.eclipse.swt.widgets.Widget::jsObject.setItemValue(index,string);
}-*/;

}
